﻿unit tb.ArrowControl;

interface

uses
  System.Classes,
  System.Types,
  System.Math,
  Vcl.Controls,
  Vcl.Graphics,
  Winapi.Windows,
  Winapi.Messages,
  Winapi.GDIPAPI,
  Winapi.GDIPOBJ,
  GDIPUTIL;

type
  TArrowPartEvent = procedure(Sender: TObject; PartIndex: Integer) of object;

  TArrowControl = class(TCustomControl)
  private
    FAngle: Single;
    FHoverPart: Integer;
    FRegions: array[0..4] of HRGN;

    FPart0: TArray<TPointF>;
    FPart1: TArray<TPointF>;
    FPart2: TArray<TPointF>;
    FPart3: TArray<TPointF>;
    FPart4: TArray<TPointF>;

    FOnClickPart: TArrowPartEvent;
    FOnHoverPart: TArrowPartEvent;

    FWidthScale: Single;
    FHeightScale: Single;
    FGap: Single;
    FColorDefault: TColor;
    FColorHover: TColor;

    FColorStroke: TColor;
    FStrokeWidth: Single;

    procedure InitGeometry;

    procedure SetAngle(const Value: Single);
    procedure SetWidthScale(const Value: Single);
    procedure SetHeightScale(const Value: Single);
    procedure SetGap(const Value: Single);
    procedure SetColorDefault(const Value: TColor);
    procedure SetColorHover(const Value: TColor);
    procedure SetColorStroke(const Value: TColor);
    procedure SetStrokeWidth(const Value: Single);

    procedure UpdateRegions;
    procedure ClearRegions;
    function PartCount: Integer;

    procedure CMMouseLeave(var Msg: TMessage); message CM_MOUSELEAVE;
  protected
    procedure Paint; override;
    procedure Resize; override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Angle: Single read FAngle write SetAngle;

    property WidthScale: Single read FWidthScale write SetWidthScale;
    property HeightScale: Single read FHeightScale write SetHeightScale;
    property Gap: Single read FGap write SetGap;

    property ColorDefault: TColor read FColorDefault write SetColorDefault;
    property ColorHover: TColor read FColorHover write SetColorHover;

    property ColorStroke: TColor read FColorStroke write SetColorStroke;
    property StrokeWidth: Single read FStrokeWidth write SetStrokeWidth;

    property OnClickPart: TArrowPartEvent read FOnClickPart write FOnClickPart;
    property OnHoverPart: TArrowPartEvent read FOnHoverPart write FOnHoverPart;

    property Align;
    property Anchors;
    property Color;
    property ParentColor;
    property ParentBackground;
    property Visible;
  end;

procedure Register;

implementation

{---------------------------------------------------------------}
{ Hilfsfunktionen                                               }
{---------------------------------------------------------------}

function DegToRadEx(AngleDeg: Single): Double;
begin
  Result := AngleDeg * Pi / 180.0;
end;

function ColorToARGB(const AColor: TColor): ARGB;
var
  C: COLORREF;
begin
  C := ColorToRGB(AColor);
  Result := MakeColor(255, GetRValue(C), GetGValue(C), GetBValue(C));
end;

function TransformPoint(const P: TPointF; const Center: TPointF;
  AngleRad: Double; ScaleX, ScaleY: Double): TPointF;
var
  x, y, cx, cy: Double;
begin
  x := P.X;
  y := P.Y - 0.5;

  x := x * ScaleX;
  y := y * ScaleY;

  cx := Cos(AngleRad);
  cy := Sin(AngleRad);

  Result.X := Center.X + x * cx - y * cy;
  Result.Y := Center.Y + x * cy + y * cx;
end;

procedure TransformPoly(const Src: array of TPointF; var Dst: TArray<TPoint>;
  const Center: TPointF; AngleRad: Double;
  ScaleX, ScaleY: Double);
var
  i: Integer;
  p: TPointF;
begin
  SetLength(Dst, Length(Src));
  for i := 0 to High(Src) do
  begin
    p := TransformPoint(Src[i], Center, AngleRad, ScaleX, ScaleY);
    Dst[i].X := Round(p.X);
    Dst[i].Y := Round(p.Y);
  end;
end;

{---------------------------------------------------------------}
{ Geometrie                                                     }
{---------------------------------------------------------------}

procedure TArrowControl.InitGeometry;
const
  Xmin = 4.620403461276595;
  Xmax = 18.64789037875039;
  Ymin = -30.39960118282205;
  Ymax = -4.050965379263002;

  W = Xmax - Xmin;
  H = Ymax - Ymin;

  function N(x, y: Double): TPointF;
  begin
    Result.X := (x - Xmin) / W - 0.5;
    Result.Y := (Ymax - y) / H;
  end;

  procedure OffsetPart(var P: TArray<TPointF>; OffsetY: Single);
  var
    i: Integer;
  begin
    for i := 0 to High(P) do
      P[i].Y := P[i].Y + OffsetY;
  end;

begin
  FPart0 := [
    N(11.66643999463558, -4.050965379263002),
    N(6.863751631149598, -13.85699991838126),
    N(16.42510543176447, -13.85699991838126)
  ];

  FPart1 := [
    N(4.620403461276595, -18.43742404173103),
    N(7.070810588714067, -18.43742404173103),
    N(7.070810588714067, -19.1908366147903),
    N(16.19351971742472, -19.1908366147903),
    N(16.19748325131292, -18.43742404173103),
    N(18.64789037875039, -18.43742404173103),
    N(16.97175344395702, -14.98346055301135),
    N(6.312046519334908, -14.98346055301135)
  ];

  FPart2 := [
    N(7.321446409199889, -20.54239756040535),
    N(7.907180907628071, -23.72894695446837),
    N(15.24966322238655, -23.72894695446837),
    N(15.91241602415286, -20.54239756040535)
  ];

  FPart3 := [
    N(8.1258551278477, -24.91859210224089),
    N(15.00223550127138, -24.91859210224089),
    N(14.56040030009384, -27.04295836494957),
    N(8.516344793466487, -27.04295836494957)
  ];

  FPart4 := [
    N(8.742828798166641, -28.27509078992868),
    N(14.30413588494826, -28.27509078992868),
    N(13.86263243718247, -30.39786196525283),
    N(9.133344957026547, -30.39960118282205)
  ];

  if FGap <> 0 then
  begin
    OffsetPart(FPart1, FGap * 1);
    OffsetPart(FPart2, FGap * 2);
    OffsetPart(FPart3, FGap * 3);
    OffsetPart(FPart4, FGap * 4);
  end;
end;

{---------------------------------------------------------------}
{ Setter                                                        }
{---------------------------------------------------------------}

procedure TArrowControl.SetAngle(const Value: Single);
begin
  if not SameValue(Value, FAngle) then
  begin
    FAngle := Value;
    UpdateRegions;
    Invalidate;
  end;
end;

procedure TArrowControl.SetWidthScale(const Value: Single);
begin
  if not SameValue(Value, FWidthScale) then
  begin
    FWidthScale := Value;
    UpdateRegions;
    Invalidate;
  end;
end;

procedure TArrowControl.SetHeightScale(const Value: Single);
begin
  if not SameValue(Value, FHeightScale) then
  begin
    FHeightScale := Value;
    UpdateRegions;
    Invalidate;
  end;
end;

procedure TArrowControl.SetGap(const Value: Single);
begin
  if not SameValue(Value, FGap) then
  begin
    FGap := Value;
    InitGeometry;
    UpdateRegions;
    Invalidate;
  end;
end;

procedure TArrowControl.SetColorDefault(const Value: TColor);
begin
  if FColorDefault <> Value then
  begin
    FColorDefault := Value;
    Invalidate;
  end;
end;

procedure TArrowControl.SetColorHover(const Value: TColor);
begin
  if FColorHover <> Value then
  begin
    FColorHover := Value;
    Invalidate;
  end;
end;

procedure TArrowControl.SetColorStroke(const Value: TColor);
begin
  if FColorStroke <> Value then
  begin
    FColorStroke := Value;
    Invalidate;
  end;
end;

procedure TArrowControl.SetStrokeWidth(const Value: Single);
begin
  if not SameValue(Value, FStrokeWidth) then
  begin
    FStrokeWidth := Value;
    Invalidate;
  end;
end;

{---------------------------------------------------------------}
{ Konstruktor / Destruktor                                      }
{---------------------------------------------------------------}

constructor TArrowControl.Create(AOwner: TComponent);
begin
  inherited;
  DoubleBuffered := True;
  FHoverPart := -1;
  FAngle := 0;
  Color := clBtnFace;

  FWidthScale := 1.0;
  FHeightScale := 1.0;
  FGap := 0.0;

  FColorDefault := clBlue;
  FColorHover := clRed;

  FColorStroke := clBlack;
  FStrokeWidth := 1.0;

  InitGeometry;
end;

destructor TArrowControl.Destroy;
begin
  ClearRegions;
  inherited;
end;

{---------------------------------------------------------------}
{ Regionen                                                      }
{---------------------------------------------------------------}

function TArrowControl.PartCount: Integer;
begin
  Result := 5;
end;

procedure TArrowControl.ClearRegions;
var
  i: Integer;
begin
  for i := 0 to PartCount - 1 do
    if FRegions[i] <> 0 then
    begin
      DeleteObject(FRegions[i]);
      FRegions[i] := 0;
    end;
end;

procedure TArrowControl.UpdateRegions;
var
  i: Integer;
  ptsI: TArray<TPoint>;
  cx, cy, s, sx, sy: Double;
  center: TPointF;
  angleRad: Double;

  function GetPart(i: Integer): TArray<TPointF>;
  begin
    case i of
      0: Result := FPart0;
      1: Result := FPart1;
      2: Result := FPart2;
      3: Result := FPart3;
      4: Result := FPart4;
    end;
  end;

begin
  ClearRegions;

  cx := Width / 2;
  cy := Height / 2;
  center := PointF(cx, cy);

  s := Min(Width, Height) * 0.9;
  sx := s * FWidthScale;
  sy := s * FHeightScale;

  angleRad := DegToRadEx(FAngle);

  for i := 0 to PartCount - 1 do
  begin
    TransformPoly(GetPart(i), ptsI, center, angleRad, sx, sy);
    if Length(ptsI) > 0 then
      FRegions[i] := CreatePolygonRgn(ptsI[0], Length(ptsI), WINDING)
    else
      FRegions[i] := 0;
  end;
end;

{---------------------------------------------------------------}
{ Resize                                                        }
{---------------------------------------------------------------}

procedure TArrowControl.Resize;
begin
  inherited;
  UpdateRegions;
end;

{---------------------------------------------------------------}
{ Paint (GDI+ Antialiasing)                                     }
{---------------------------------------------------------------}

procedure TArrowControl.Paint;
var
  i: Integer;
  ptsI: TArray<TPoint>;
  cx, cy, s, sx, sy: Double;
  center: TPointF;
  angleRad: Double;

  G: TGPGraphics;
  GPBrush: TGPSolidBrush;
  GPPen: TGPPen;
  GPPoints: array of TGPPoint;
  j: Integer;

  function GetPart(i: Integer): TArray<TPointF>;
  begin
    case i of
      0: Result := FPart0;
      1: Result := FPart1;
      2: Result := FPart2;
      3: Result := FPart3;
      4: Result := FPart4;
    end;
  end;

begin
  inherited;

  Canvas.Brush.Color := Color;
  Canvas.FillRect(ClientRect);

  cx := Width / 2;
  cy := Height / 2;
  center := PointF(cx, cy);
  s := Min(Width, Height) * 0.9;
  sx := s * FWidthScale;
  sy := s * FHeightScale;
  angleRad := DegToRadEx(FAngle);

  G := TGPGraphics.Create(Canvas.Handle);
  try
    G.SetSmoothingMode(SmoothingModeAntiAlias);

    for i := 0 to PartCount - 1 do
    begin
      TransformPoly(GetPart(i), ptsI, center, angleRad, sx, sy);

      SetLength(GPPoints, Length(ptsI));
      for j := 0 to High(ptsI) do
        GPPoints[j] := MakePoint(ptsI[j].X, ptsI[j].Y);

      if i = FHoverPart then
        GPBrush := TGPSolidBrush.Create(ColorToARGB(FColorHover))
      else
        GPBrush := TGPSolidBrush.Create(ColorToARGB(FColorDefault));

      GPPen := TGPPen.Create(ColorToARGB(FColorStroke), FStrokeWidth);

      if Length(GPPoints) > 0 then
      begin
        G.FillPolygon(GPBrush, PGPPoint(@GPPoints[0]), Length(GPPoints));
        G.DrawPolygon(GPPen, PGPPoint(@GPPoints[0]), Length(GPPoints));
      end;

      GPBrush.Free;
      GPPen.Free;
    end;

  finally
    G.Free;
  end;
end;

{---------------------------------------------------------------}
{ Maus                                                          }
{---------------------------------------------------------------}

procedure TArrowControl.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  i, hit: Integer;
begin
  inherited;
  hit := -1;

  for i := 0 to PartCount - 1 do
    if (FRegions[i] <> 0) and PtInRegion(FRegions[i], X, Y) then
    begin
      hit := i;
      Break;
    end;

  if hit <> FHoverPart then
  begin
    FHoverPart := hit;
    Invalidate;
    if Assigned(FOnHoverPart) then
      FOnHoverPart(Self, FHoverPart);
  end;
end;

procedure TArrowControl.CMMouseLeave(var Msg: TMessage);
begin
  inherited;
  if FHoverPart <> -1 then
  begin
    FHoverPart := -1;
    Invalidate;
    if Assigned(FOnHoverPart) then
      FOnHoverPart(Self, -1);
  end;
end;

procedure TArrowControl.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  i: Integer;
begin
  inherited;
  if Button = mbLeft then
    for i := 0 to PartCount - 1 do
      if (FRegions[i] <> 0) and PtInRegion(FRegions[i], X, Y) then
      begin
        if Assigned(FOnClickPart) then
          FOnClickPart(Self, i);
        Break;
      end;
end;

{---------------------------------------------------------------}
{ Register                                                      }
{---------------------------------------------------------------}

procedure Register;
begin
  RegisterComponents('TOBY Software', [TArrowControl]);
end;

end.

